<?php
declare(strict_types=1);

require_once dirname(__DIR__) . '/config/config.php';
require_once CLASSES_PATH . '/JWT.php';
require_once CLASSES_PATH . '/Database.php';

/**
 * ============================================================
 * FRESIL C.A. — Middleware de Autenticación
 * ─────────────────────────────────────────────────────────
 * Incluir al inicio de cualquier endpoint protegido:
 *
 *   require_once ROOT_PATH . '/middleware/AuthMiddleware.php';
 *   $user = AuthMiddleware::requireAuth();
 *   // $user = ['id', 'email', 'nombre', 'rol', ...]
 *
 * Para requerir un rol específico:
 *   AuthMiddleware::requireRole('admin');
 *   AuthMiddleware::requireRole(['admin', 'almacen']);
 * ============================================================
 */
class AuthMiddleware
{
    /**
     * Verificar token JWT y retornar payload del usuario.
     * Detiene la ejecución con 401 si no está autenticado.
     */
    public static function requireAuth(): array
    {
        $authHeader = $_SERVER['HTTP_AUTHORIZATION']
            ?? $_SERVER['REDIRECT_HTTP_AUTHORIZATION']
            ?? '';

        // Soporte para Apache que puede eliminar el header Authorization
        if (empty($authHeader) && function_exists('apache_request_headers')) {
            $headers    = apache_request_headers();
            $authHeader = $headers['Authorization'] ?? $headers['authorization'] ?? '';
        }

        if (empty($authHeader)) {
            self::abort(401, 'Token de autenticación requerido.');
        }

        try {
            if (!preg_match('/^Bearer\s+(.+)$/i', trim($authHeader), $matches)) {
                self::abort(401, 'Formato de token inválido. Use: Bearer <token>');
            }

            $payload = JWT::validate($matches[1]);

            // Verificar que el usuario aún existe y está activo en BD
            $db      = Database::getInstance();
            $usuario = $db->fetchOne(
                'SELECT id, nombre, email, rol, activo FROM usuarios WHERE id = :id AND activo = 1 LIMIT 1',
                [':id' => (int)$payload['sub']]
            );

            if (!$usuario) {
                self::abort(401, 'Usuario no encontrado o cuenta desactivada.');
            }

            return $usuario;

        } catch (RuntimeException $e) {
            self::abort($e->getCode() ?: 401, $e->getMessage());
        }
    }

    /**
     * Verificar rol(es) requerido(s). Llama requireAuth() internamente.
     *
     * @param string|array $roles Rol o arreglo de roles permitidos
     * @return array Datos del usuario autenticado
     */
    public static function requireRole(string|array $roles): array
    {
        $usuario = self::requireAuth();

        $rolesPermitidos = is_array($roles) ? $roles : [$roles];

        if (!in_array($usuario['rol'], $rolesPermitidos, true)) {
            self::abort(403, 'Acceso denegado. Rol requerido: ' . implode(' o ', $rolesPermitidos));
        }

        return $usuario;
    }

    /**
     * Obtener IP real del cliente (considerando proxies)
     */
    public static function getClientIp(): string
    {
        $headers = [
            'HTTP_CF_CONNECTING_IP',    // Cloudflare
            'HTTP_X_FORWARDED_FOR',     // Proxy
            'HTTP_X_REAL_IP',           // Nginx
            'REMOTE_ADDR',
        ];

        foreach ($headers as $header) {
            if (!empty($_SERVER[$header])) {
                $ips = explode(',', $_SERVER[$header]);
                $ip  = trim($ips[0]);
                if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE)) {
                    return $ip;
                }
            }
        }

        return $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
    }

    /**
     * Respuesta de error JSON y detener ejecución
     */
    public static function abort(int $code, string $message): never
    {
        http_response_code($code);
        header('Content-Type: application/json; charset=utf-8');
        echo json_encode([
            'ok'      => false,
            'codigo'  => $code,
            'mensaje' => $message,
        ], JSON_UNESCAPED_UNICODE);
        exit;
    }

    /**
     * Respuesta JSON de éxito estandarizada
     */
    public static function respond(mixed $data, int $code = 200): never
    {
        http_response_code($code);
        header('Content-Type: application/json; charset=utf-8');
        echo json_encode([
            'ok'   => true,
            'data' => $data,
        ], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
        exit;
    }
}
