-- ============================================================
-- FRESIL C.A. — SQL Módulo 02: Catálogo de Repuestos
-- Ejecutar DESPUÉS de 001_auth_schema.sql
-- Compatible con MySQL 8.0 / MariaDB 10.11
-- ============================================================

USE fresil_inventario;

-- ─────────────────────────────────────────────────────────
-- TABLA: tasas_cambio (si no existe del M01)
-- ─────────────────────────────────────────────────────────
CREATE TABLE IF NOT EXISTS tasas_cambio (
  id             INT UNSIGNED    NOT NULL AUTO_INCREMENT,
  fecha          DATE            NOT NULL,
  tasa_bcv       DECIMAL(12,4)   NOT NULL COMMENT 'Tasa Banco Central Venezuela',
  tasa_paralela  DECIMAL(12,4)   NULL     COMMENT 'Tasa mercado paralelo',
  tasa_usada     DECIMAL(12,4)   NOT NULL COMMENT 'Tasa que usa el negocio',
  usuario_id     INT UNSIGNED    NOT NULL,
  created_at     TIMESTAMP       NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (id),
  UNIQUE KEY uq_fecha (fecha),
  KEY idx_fecha (fecha),
  CONSTRAINT fk_tasa_usr FOREIGN KEY (usuario_id) REFERENCES usuarios(id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Tasa inicial de referencia
INSERT IGNORE INTO tasas_cambio (fecha, tasa_bcv, tasa_paralela, tasa_usada, usuario_id)
VALUES (CURDATE(), 36.50, 38.00, 38.00, 1);

-- ─────────────────────────────────────────────────────────
-- TABLA: categorias_repuesto (árbol recursivo con parent_id)
-- ─────────────────────────────────────────────────────────
CREATE TABLE IF NOT EXISTS categorias_repuesto (
  id           INT UNSIGNED    NOT NULL AUTO_INCREMENT,
  parent_id    INT UNSIGNED    NULL     COMMENT 'NULL = categoría raíz',
  nombre       VARCHAR(100)    NOT NULL,
  descripcion  VARCHAR(255)    NULL,
  icono        VARCHAR(10)     NULL     COMMENT 'Emoji representativo',
  activo       TINYINT(1)      NOT NULL DEFAULT 1,
  orden        SMALLINT        NOT NULL DEFAULT 0,
  PRIMARY KEY (id),
  KEY idx_parent (parent_id),
  KEY idx_activo (activo),
  KEY idx_orden  (orden),
  CONSTRAINT fk_cat_parent FOREIGN KEY (parent_id)
    REFERENCES categorias_repuesto(id) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Categorías base para taller mecánico venezolano
INSERT IGNORE INTO categorias_repuesto (id, parent_id, nombre, icono, orden) VALUES
  (1,  NULL, 'Motor',               '⚙️',  1),
  (2,  NULL, 'Sistema de Frenos',   '🔴',  2),
  (3,  NULL, 'Suspensión',          '🔧',  3),
  (4,  NULL, 'Dirección',           '🎯',  4),
  (5,  NULL, 'Sistema Eléctrico',   '⚡',  5),
  (6,  NULL, 'Silenciadores',       '🔇',  6),
  (7,  NULL, 'Transmisión',         '⚙️',  7),
  (8,  NULL, 'Refrigeración',       '🌡️', 8),
  (9,  NULL, 'Combustible',         '⛽',  9),
  (10, NULL, 'Carrocería',          '🚗',  10),
  (11, NULL, 'Aceites y Fluidos',   '🛢️', 11),
  (12, NULL, 'Filtros',             '🔲',  12),
  -- Sub-categorías Motor
  (13, 1, 'Pistones y Cilindros',   NULL, 1),
  (14, 1, 'Distribución',           NULL, 2),
  (15, 1, 'Válvulas y Culata',      NULL, 3),
  (16, 1, 'Cigüeñal y Bielas',      NULL, 4),
  -- Sub-categorías Frenos
  (17, 2, 'Pastillas de Freno',     NULL, 1),
  (18, 2, 'Discos de Freno',        NULL, 2),
  (19, 2, 'Tambores',               NULL, 3),
  (20, 2, 'Bombas de Freno',        NULL, 4),
  (21, 2, 'Mangueras de Freno',     NULL, 5),
  -- Sub-categorías Suspensión
  (22, 3, 'Amortiguadores',         NULL, 1),
  (23, 3, 'Resortes',               NULL, 2),
  (24, 3, 'Bujes y Rotulas',        NULL, 3),
  (25, 3, 'Bases de Amortiguador',  NULL, 4),
  -- Sub-categorías Silenciadores
  (26, 6, 'Tubos de Escape',        NULL, 1),
  (27, 6, 'Catalizadores',          NULL, 2),
  (28, 6, 'Silenciadores Traseros', NULL, 3),
  -- Sub-categorías Eléctrico
  (29, 5, 'Baterías',               NULL, 1),
  (30, 5, 'Alternadores',           NULL, 2),
  (31, 5, 'Arranques',              NULL, 3),
  (32, 5, 'Sensores',               NULL, 4),
  -- Sub-categorías Filtros
  (33, 12, 'Filtro de Aceite',      NULL, 1),
  (34, 12, 'Filtro de Aire',        NULL, 2),
  (35, 12, 'Filtro de Combustible', NULL, 3),
  (36, 12, 'Filtro de Habitáculo',  NULL, 4);

-- ─────────────────────────────────────────────────────────
-- TABLA: marcas_vehiculo
-- ─────────────────────────────────────────────────────────
CREATE TABLE IF NOT EXISTS marcas_vehiculo (
  id           INT UNSIGNED    NOT NULL AUTO_INCREMENT,
  nombre       VARCHAR(80)     NOT NULL,
  pais_origen  VARCHAR(50)     NULL,
  activo       TINYINT(1)      NOT NULL DEFAULT 1,
  PRIMARY KEY (id),
  UNIQUE KEY uq_marca (nombre),
  KEY idx_activo (activo)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Marcas más comunes en Venezuela
INSERT IGNORE INTO marcas_vehiculo (nombre, pais_origen) VALUES
  ('Toyota',        'Japón'),
  ('Chevrolet',     'Estados Unidos'),
  ('Ford',          'Estados Unidos'),
  ('Hyundai',       'Corea del Sur'),
  ('Kia',           'Corea del Sur'),
  ('Mitsubishi',    'Japón'),
  ('Nissan',        'Japón'),
  ('Honda',         'Japón'),
  ('Volkswagen',    'Alemania'),
  ('Renault',       'Francia'),
  ('Dodge',         'Estados Unidos'),
  ('Jeep',          'Estados Unidos'),
  ('Mazda',         'Japón'),
  ('Suzuki',        'Japón'),
  ('Fiat',          'Italia'),
  ('Chery',         'China'),
  ('BAIC',          'China'),
  ('Mack',          'Estados Unidos'),
  ('Mercedes-Benz', 'Alemania'),
  ('Iveco',         'Italia');

-- ─────────────────────────────────────────────────────────
-- TABLA: modelos_vehiculo
-- ─────────────────────────────────────────────────────────
CREATE TABLE IF NOT EXISTS modelos_vehiculo (
  id        INT UNSIGNED    NOT NULL AUTO_INCREMENT,
  marca_id  INT UNSIGNED    NOT NULL,
  nombre    VARCHAR(100)    NOT NULL,
  tipo      ENUM('sedan','suv','pickup','van','coupe','hatchback','camioneta','bus','moto','otro')
            NOT NULL DEFAULT 'sedan',
  activo    TINYINT(1)      NOT NULL DEFAULT 1,
  PRIMARY KEY (id),
  UNIQUE KEY uq_marca_modelo (marca_id, nombre),
  KEY idx_marca  (marca_id),
  KEY idx_activo (activo),
  CONSTRAINT fk_modelo_marca FOREIGN KEY (marca_id)
    REFERENCES marcas_vehiculo(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Modelos comunes en Venezuela
INSERT IGNORE INTO modelos_vehiculo (marca_id, nombre, tipo) VALUES
  -- Toyota (id=1)
  (1, 'Corolla',        'sedan'),
  (1, 'Camry',          'sedan'),
  (1, 'Yaris',          'hatchback'),
  (1, 'Land Cruiser',   'suv'),
  (1, 'Prado',          'suv'),
  (1, 'Hilux',          'pickup'),
  (1, 'RAV4',           'suv'),
  (1, 'Fortuner',       'suv'),
  -- Chevrolet (id=2)
  (2, 'Aveo',           'sedan'),
  (2, 'Spark',          'hatchback'),
  (2, 'Cruze',          'sedan'),
  (2, 'Captiva',        'suv'),
  (2, 'Silverado',      'pickup'),
  (2, 'Tahoe',          'suv'),
  -- Ford (id=3)
  (3, 'Explorer',       'suv'),
  (3, 'F-150',          'pickup'),
  (3, 'Focus',          'sedan'),
  (3, 'Ranger',         'pickup'),
  (3, 'Escape',         'suv'),
  -- Hyundai (id=4)
  (4, 'Accent',         'sedan'),
  (4, 'Tucson',         'suv'),
  (4, 'Santa Fe',       'suv'),
  (4, 'Elantra',        'sedan'),
  -- Kia (id=5)
  (5, 'Rio',            'sedan'),
  (5, 'Sportage',       'suv'),
  (5, 'Sorento',        'suv'),
  (5, 'Cerato',         'sedan'),
  -- Mitsubishi (id=6)
  (6, 'L200',           'pickup'),
  (6, 'Montero',        'suv'),
  (6, 'Outlander',      'suv'),
  (6, 'Lancer',         'sedan'),
  -- Nissan (id=7)
  (7, 'Sentra',         'sedan'),
  (7, 'Frontier',       'pickup'),
  (7, 'Pathfinder',     'suv'),
  (7, 'Patrol',         'suv'),
  (7, 'Versa',          'sedan'),
  -- Honda (id=8)
  (8, 'Civic',          'sedan'),
  (8, 'Accord',         'sedan'),
  (8, 'CR-V',           'suv'),
  (8, 'Pilot',          'suv'),
  -- Dodge (id=11)
  (11,'Ram 1500',       'pickup'),
  (11,'Journey',        'suv'),
  -- Jeep (id=12)
  (12,'Cherokee',       'suv'),
  (12,'Grand Cherokee', 'suv'),
  (12,'Wrangler',       'suv');

-- ─────────────────────────────────────────────────────────
-- TABLA: vehiculo_anios (rango de años por versión/motor)
-- ─────────────────────────────────────────────────────────
CREATE TABLE IF NOT EXISTS vehiculo_anios (
  id           INT UNSIGNED    NOT NULL AUTO_INCREMENT,
  modelo_id    INT UNSIGNED    NOT NULL,
  anio_inicio  YEAR            NOT NULL,
  anio_fin     YEAR            NULL     COMMENT 'NULL = modelo vigente',
  motor_cc     VARCHAR(20)     NULL     COMMENT 'Ej: 1.6, 2.0T, 3.5 V6',
  combustible  ENUM('gasolina','diesel','hibrido','electrico') NOT NULL DEFAULT 'gasolina',
  PRIMARY KEY (id),
  KEY idx_modelo (modelo_id),
  KEY idx_anios  (anio_inicio, anio_fin),
  CONSTRAINT fk_va_modelo FOREIGN KEY (modelo_id)
    REFERENCES modelos_vehiculo(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Rangos de años representativos
INSERT IGNORE INTO vehiculo_anios (modelo_id, anio_inicio, anio_fin, motor_cc) VALUES
  -- Corolla (1) - múltiples generaciones
  (1, 2000, 2006, '1.6'), (1, 2007, 2013, '1.8'), (1, 2014, 2019, '1.8'),
  (1, 2020, NULL, '2.0'),
  -- Hilux (6)
  (6, 2005, 2011, '2.7'), (6, 2005, 2011, '3.0 Diesel'),
  (6, 2012, 2015, '2.7'), (6, 2016, NULL, '2.8 Diesel'),
  -- Aveo (9)
  (9, 2005, 2011, '1.4'), (9, 2005, 2011, '1.6'),
  (9, 2012, 2017, '1.4'), (9, 2018, NULL, '1.4'),
  -- L200 (29)
  (29, 2006, 2014, '2.5 Diesel'), (29, 2015, NULL, '2.4 Diesel'),
  -- Sentra (33)
  (33, 2000, 2006, '1.8'), (33, 2007, 2012, '2.0'), (33, 2013, NULL, '1.8'),
  -- Civic (41)
  (41, 2001, 2005, '1.7'), (41, 2006, 2011, '1.8'), (41, 2012, 2015, '1.8'),
  (41, 2016, NULL, '2.0');

-- ─────────────────────────────────────────────────────────
-- TABLA: repuestos (CORE del módulo)
-- ─────────────────────────────────────────────────────────
CREATE TABLE IF NOT EXISTS repuestos (
  id                   BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
  codigo_interno       VARCHAR(50)     NOT NULL COMMENT 'Código propio del taller',
  codigo_barras        VARCHAR(60)     NULL     COMMENT 'EAN-13, Code128, UPC-A',
  codigo_oem           VARCHAR(100)    NULL     COMMENT 'Código fabricante original',
  categoria_id         INT UNSIGNED    NOT NULL,
  proveedor_id         BIGINT UNSIGNED NULL     COMMENT 'Proveedor principal (NULL hasta M04)',
  nombre               VARCHAR(200)    NOT NULL,
  descripcion          TEXT            NULL,
  marca_repuesto       VARCHAR(80)     NULL     COMMENT 'Ej: NGK, Bosch, Brembo, Monroe',
  unidad_medida        ENUM('unidad','par','juego','litro','metro','kilo','set') NOT NULL DEFAULT 'unidad',
  precio_compra_usd    DECIMAL(12,4)   NOT NULL DEFAULT 0.0000,
  precio_venta_usd     DECIMAL(12,4)   NOT NULL DEFAULT 0.0000,
  precio_compra_bs     DECIMAL(18,4)   NOT NULL DEFAULT 0.0000,
  precio_venta_bs      DECIMAL(18,4)   NOT NULL DEFAULT 0.0000,
  tasa_cambio          DECIMAL(12,4)   NOT NULL DEFAULT 0.0000,
  margen_ganancia_pct  DECIMAL(6,2)    NOT NULL DEFAULT 0.00,
  stock_actual         INT             NOT NULL DEFAULT 0,
  stock_minimo         INT             NOT NULL DEFAULT 1,
  stock_maximo         INT             NULL,
  ubicacion_almacen    VARCHAR(30)     NULL     COMMENT 'Ej: A-3-2 (Pasillo-Estante-Nivel)',
  imagen_url           VARCHAR(500)    NULL,
  activo               TINYINT(1)      NOT NULL DEFAULT 1,
  created_at           TIMESTAMP       NOT NULL DEFAULT CURRENT_TIMESTAMP,
  updated_at           TIMESTAMP       NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  created_by           INT UNSIGNED    NULL,
  PRIMARY KEY (id),
  UNIQUE KEY uq_codigo_interno  (codigo_interno),
  UNIQUE KEY uq_codigo_barras   (codigo_barras),
  KEY idx_categoria    (categoria_id),
  KEY idx_proveedor    (proveedor_id),
  KEY idx_stock_bajo   (stock_actual, stock_minimo),
  KEY idx_activo       (activo),
  KEY idx_created      (created_at),
  FULLTEXT KEY ft_repuesto (nombre, descripcion, codigo_oem, marca_repuesto),
  CONSTRAINT fk_rep_cat FOREIGN KEY (categoria_id)
    REFERENCES categorias_repuesto(id),
  CONSTRAINT fk_rep_creador FOREIGN KEY (created_by)
    REFERENCES usuarios(id) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ─────────────────────────────────────────────────────────
-- TABLA: repuesto_compatibilidad (N:M repuesto ↔ vehículo_año)
-- ─────────────────────────────────────────────────────────
CREATE TABLE IF NOT EXISTS repuesto_compatibilidad (
  id                BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
  repuesto_id       BIGINT UNSIGNED NOT NULL,
  vehiculo_anio_id  INT UNSIGNED    NOT NULL,
  posicion          VARCHAR(50)     NULL COMMENT 'Ej: delantero izq, trasero der',
  notas             VARCHAR(255)    NULL,
  PRIMARY KEY (id),
  UNIQUE KEY uq_compat (repuesto_id, vehiculo_anio_id),
  KEY idx_repuesto (repuesto_id),
  KEY idx_vehiculo (vehiculo_anio_id),
  CONSTRAINT fk_rc_repuesto FOREIGN KEY (repuesto_id)
    REFERENCES repuestos(id) ON DELETE CASCADE,
  CONSTRAINT fk_rc_vehiculo FOREIGN KEY (vehiculo_anio_id)
    REFERENCES vehiculo_anios(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ─────────────────────────────────────────────────────────
-- VISTA: repuestos con stock bajo (para alertas)
-- ─────────────────────────────────────────────────────────
CREATE OR REPLACE VIEW v_stock_bajo AS
SELECT
  r.id, r.codigo_interno, r.codigo_barras, r.nombre,
  r.stock_actual, r.stock_minimo,
  (r.stock_minimo - r.stock_actual) AS deficit,
  c.nombre AS categoria
FROM repuestos r
JOIN categorias_repuesto c ON c.id = r.categoria_id
WHERE r.stock_actual <= r.stock_minimo AND r.activo = 1
ORDER BY deficit DESC;

-- ─────────────────────────────────────────────────────────
-- VISTA: búsqueda repuesto ↔ vehículo (YMME)
-- ─────────────────────────────────────────────────────────
CREATE OR REPLACE VIEW v_repuesto_vehiculo AS
SELECT
  r.id          AS repuesto_id,
  r.codigo_interno,
  r.codigo_barras,
  r.nombre      AS repuesto,
  r.precio_venta_usd,
  r.precio_venta_bs,
  r.stock_actual,
  mv.id         AS marca_id,
  mv.nombre     AS marca,
  mo.id         AS modelo_id,
  mo.nombre     AS modelo,
  va.id         AS vehiculo_anio_id,
  va.anio_inicio,
  va.anio_fin,
  va.motor_cc,
  rc.posicion
FROM repuesto_compatibilidad rc
JOIN repuestos        r  ON r.id  = rc.repuesto_id
JOIN vehiculo_anios   va ON va.id = rc.vehiculo_anio_id
JOIN modelos_vehiculo mo ON mo.id = va.modelo_id
JOIN marcas_vehiculo  mv ON mv.id = mo.marca_id
WHERE r.activo = 1;

-- ─────────────────────────────────────────────────────────
-- ÍNDICE AUTOMÁTICO: Secuencia para código interno
-- ─────────────────────────────────────────────────────────
CREATE TABLE IF NOT EXISTS secuencias (
  nombre    VARCHAR(50)    NOT NULL,
  siguiente INT UNSIGNED   NOT NULL DEFAULT 1,
  PRIMARY KEY (nombre)
) ENGINE=InnoDB;

INSERT IGNORE INTO secuencias (nombre, siguiente) VALUES ('repuesto_codigo', 1001);

-- ─────────────────────────────────────────────────────────
-- DATOS DE MUESTRA (10 repuestos iniciales)
-- ─────────────────────────────────────────────────────────
INSERT IGNORE INTO repuestos
  (codigo_interno, codigo_barras, codigo_oem, categoria_id, nombre, marca_repuesto,
   unidad_medida, precio_compra_usd, precio_venta_usd, precio_compra_bs, precio_venta_bs,
   tasa_cambio, margen_ganancia_pct, stock_actual, stock_minimo, ubicacion_almacen, created_by)
VALUES
  ('FRS-1001','7891234560001','04465-02170', 17,'Pastillas de Freno Delanteras Corolla',
   'Brembo','par', 18.00,28.00, 684.00,1064.00, 38.00,55.56, 8,2,'A-1-1',1),
  ('FRS-1002','7891234560002','42410-02170', 22,'Amortiguador Delantero Hilux',
   'Monroe','unidad',35.00,55.00,1330.00,2090.00,38.00,57.14,4,2,'B-2-1',1),
  ('FRS-1003','7891234560003','90915-YZZD3', 33,'Filtro de Aceite Toyota Corolla/Yaris',
   'Toyota OEM','unidad',3.50,7.00,133.00,266.00,38.00,100.00,25,5,'C-1-2',1),
  ('FRS-1004','7891234560004','17801-31090', 34,'Filtro de Aire Toyota Corolla 2014+',
   'Mann','unidad',6.00,12.00,228.00,456.00,38.00,100.00,15,3,'C-1-3',1),
  ('FRS-1005','7891234560005','NGK-BKR5EGP', 14,'Bujías NGK Platinum Corolla/Aveo',
   'NGK','juego',12.00,22.00,456.00,836.00,38.00,83.33,20,4,'D-1-1',1),
  ('FRS-1006','7891234560006','19045-74050', 8,'Termostato Toyota Corolla 1.8',
   'Gates','unidad',8.00,16.00,304.00,608.00,38.00,100.00,6,2,'E-2-1',1),
  ('FRS-1007','7891234560007','45503-09260', 4,'Terminal de Dirección Toyota Corolla Der',
   'Moog','unidad',14.00,26.00,532.00,988.00,38.00,85.71,5,2,'B-3-2',1),
  ('FRS-1008','7891234560008','43512-02220', 18,'Disco de Freno Delantero Corolla 2014+',
   'Brembo','par',45.00,75.00,1710.00,2850.00,38.00,66.67,3,2,'A-1-2',1),
  ('FRS-1009','7891234560009','04466-YZZF8', 17,'Pastillas de Freno Traseras Hilux',
   'Brembo','par',20.00,32.00,760.00,1216.00,38.00,60.00,6,2,'A-2-1',1),
  ('FRS-1010','7891234560010','33501-26020', 11,'Aceite de Motor 20W50 Mineral 1L',
   'Mobil','litro',4.50,9.00,171.00,342.00,38.00,100.00,40,10,'F-1-1',1);

-- Compatibilidades de muestra
-- FRS-1001 pastillas Corolla → Corolla 2014-2019 1.8
INSERT IGNORE INTO repuesto_compatibilidad (repuesto_id, vehiculo_anio_id, posicion)
SELECT r.id, va.id, 'Delantero'
FROM repuestos r, vehiculo_anios va
JOIN modelos_vehiculo mo ON mo.id = va.modelo_id
JOIN marcas_vehiculo  mv ON mv.id = mo.marca_id
WHERE r.codigo_interno = 'FRS-1001'
  AND mv.nombre = 'Toyota' AND mo.nombre = 'Corolla'
  AND va.anio_inicio = 2014
LIMIT 1;

-- FRS-1003 filtro aceite → Corolla 2014-2019
INSERT IGNORE INTO repuesto_compatibilidad (repuesto_id, vehiculo_anio_id)
SELECT r.id, va.id
FROM repuestos r, vehiculo_anios va
JOIN modelos_vehiculo mo ON mo.id = va.modelo_id
JOIN marcas_vehiculo  mv ON mv.id = mo.marca_id
WHERE r.codigo_interno = 'FRS-1003'
  AND mv.nombre = 'Toyota' AND mo.nombre = 'Corolla'
LIMIT 3;

SELECT 'Módulo 02 instalado correctamente ✓' AS resultado;
SELECT COUNT(*) AS total_categorias FROM categorias_repuesto;
SELECT COUNT(*) AS total_marcas FROM marcas_vehiculo;
SELECT COUNT(*) AS total_modelos FROM modelos_vehiculo;
SELECT COUNT(*) AS total_repuestos FROM repuestos;
