-- ============================================================
-- FRESIL C.A. — SQL Módulo 01: Autenticación y Usuarios
-- Ejecutar ANTES de usar el sistema
-- Compatible con MySQL 8.0 / MariaDB 10.11
-- ============================================================

CREATE DATABASE IF NOT EXISTS fresil_inventario
  CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;

USE fresil_inventario;

-- ─────────────────────────────────────────────────────────
-- TABLA: usuarios
-- ─────────────────────────────────────────────────────────
CREATE TABLE IF NOT EXISTS usuarios (
  id             INT UNSIGNED     NOT NULL AUTO_INCREMENT,
  nombre         VARCHAR(100)     NOT NULL,
  email          VARCHAR(150)     NOT NULL,
  password_hash  VARCHAR(255)     NOT NULL,
  rol            ENUM('admin','almacen','vendedor','solo_lectura') NOT NULL DEFAULT 'vendedor',
  activo         TINYINT(1)       NOT NULL DEFAULT 1,
  ultimo_login   TIMESTAMP        NULL,
  created_at     TIMESTAMP        NOT NULL DEFAULT CURRENT_TIMESTAMP,
  updated_at     TIMESTAMP        NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (id),
  UNIQUE KEY uq_email (email),
  KEY idx_activo (activo),
  KEY idx_rol (rol)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ─────────────────────────────────────────────────────────
-- TABLA: refresh_tokens
-- ─────────────────────────────────────────────────────────
CREATE TABLE IF NOT EXISTS refresh_tokens (
  id          BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
  usuario_id  INT UNSIGNED    NOT NULL,
  token_hash  VARCHAR(128)    NOT NULL COMMENT 'SHA-256 del token, nunca texto plano',
  ip_address  VARCHAR(45)     NULL,
  revocado    TINYINT(1)      NOT NULL DEFAULT 0,
  revocado_en TIMESTAMP       NULL,
  expira_en   TIMESTAMP       NOT NULL,
  creado_en   TIMESTAMP       NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (id),
  UNIQUE KEY uq_token_hash (token_hash),
  KEY idx_usuario (usuario_id),
  KEY idx_expira (expira_en),
  KEY idx_revocado (revocado),
  CONSTRAINT fk_rt_usuario FOREIGN KEY (usuario_id) REFERENCES usuarios(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- ─────────────────────────────────────────────────────────
-- TABLA: login_intentos (rate limiting)
-- ─────────────────────────────────────────────────────────
CREATE TABLE IF NOT EXISTS login_intentos (
  id          BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
  ip_address  VARCHAR(45)     NOT NULL,
  email       VARCHAR(150)    NOT NULL,
  exitoso     TINYINT(1)      NOT NULL DEFAULT 0,
  created_at  TIMESTAMP       NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (id),
  KEY idx_ip_email (ip_address, email),
  KEY idx_created (created_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Auto-limpiar intentos viejos (evento MySQL)
CREATE EVENT IF NOT EXISTS evt_limpiar_intentos_login
  ON SCHEDULE EVERY 1 HOUR
  DO DELETE FROM login_intentos WHERE created_at < DATE_SUB(NOW(), INTERVAL 2 HOUR);

-- ─────────────────────────────────────────────────────────
-- TABLA: auditoria_log
-- ─────────────────────────────────────────────────────────
CREATE TABLE IF NOT EXISTS auditoria_log (
  id              BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
  usuario_id      INT UNSIGNED    NULL,
  tabla_afectada  VARCHAR(60)     NOT NULL,
  registro_id     BIGINT UNSIGNED NOT NULL,
  accion          VARCHAR(30)     NOT NULL,
  datos_antes     JSON            NULL,
  datos_despues   JSON            NULL,
  ip_address      VARCHAR(45)     NULL,
  user_agent      VARCHAR(300)    NULL,
  created_at      TIMESTAMP       NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (id),
  KEY idx_tabla (tabla_afectada, registro_id),
  KEY idx_usuario (usuario_id),
  KEY idx_fecha (created_at),
  KEY idx_accion (accion)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- ─────────────────────────────────────────────────────────
-- USUARIO ADMINISTRADOR INICIAL
-- ─────────────────────────────────────────────────────────
-- Contraseña: Fresil2024!  (CAMBIAR INMEDIATAMENTE)
-- Hash bcrypt generado con cost:12
INSERT IGNORE INTO usuarios (nombre, email, password_hash, rol, activo)
VALUES (
  'Administrador',
  'admin@fresil.com',
  '$2y$12$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi',
  'admin',
  1
);

-- ─────────────────────────────────────────────────────────
-- CREAR USUARIO MySQL CON MÍNIMOS PRIVILEGIOS
-- (ejecutar como root de MySQL)
-- ─────────────────────────────────────────────────────────
-- CREATE USER 'fresil_user'@'127.0.0.1' IDENTIFIED BY 'TU_PASSWORD_SEGURO_AQUI';
-- GRANT SELECT, INSERT, UPDATE ON fresil_inventario.* TO 'fresil_user'@'127.0.0.1';
-- REVOKE UPDATE ON fresil_inventario.auditoria_log FROM 'fresil_user'@'127.0.0.1';
-- REVOKE DELETE ON fresil_inventario.auditoria_log FROM 'fresil_user'@'127.0.0.1';
-- FLUSH PRIVILEGES;
