<?php
declare(strict_types=1);

require_once CLASSES_PATH . '/Database.php';

/**
 * ============================================================
 * FRESIL C.A. — Clase Usuario (Modelo)
 * ─────────────────────────────────────────────────────────
 * CRUD completo de usuarios con:
 * • Validación estricta de inputs
 * • Hash bcrypt cost:12
 * • Auditoría de cambios (antes/después)
 * • Paginación eficiente
 * ============================================================
 */
class Usuario
{
    private Database $db;

    private const ROLES_VALIDOS = ['admin', 'almacen', 'vendedor', 'solo_lectura'];
    private const CAMPOS_LISTADO = 'id, nombre, email, rol, activo, ultimo_login, created_at';

    public function __construct()
    {
        $this->db = Database::getInstance();
    }

    /**
     * Listar usuarios con paginación y filtros
     */
    public function listar(
        int    $pagina  = 1,
        int    $porPagina = 20,
        string $buscar  = '',
        string $rol     = '',
        int    $activo  = -1   // -1 = todos, 0 = inactivos, 1 = activos
    ): array {
        $pagina    = max(1, $pagina);
        $porPagina = min(100, max(5, $porPagina));
        $offset    = ($pagina - 1) * $porPagina;

        $where  = ['1=1'];
        $params = [];

        if (!empty($buscar)) {
            $where[]          = '(nombre LIKE :buscar OR email LIKE :buscar)';
            $params[':buscar'] = '%' . $buscar . '%';
        }

        if (!empty($rol) && in_array($rol, self::ROLES_VALIDOS, true)) {
            $where[]    = 'rol = :rol';
            $params[':rol'] = $rol;
        }

        if ($activo !== -1) {
            $where[]       = 'activo = :activo';
            $params[':activo'] = (int)$activo;
        }

        $whereStr = implode(' AND ', $where);

        $total = (int)$this->db->fetchOne(
            "SELECT COUNT(*) as total FROM usuarios WHERE {$whereStr}",
            $params
        )['total'];

        $params[':limit']  = $porPagina;
        $params[':offset'] = $offset;

        $usuarios = $this->db->fetchAll(
            "SELECT " . self::CAMPOS_LISTADO . "
             FROM usuarios
             WHERE {$whereStr}
             ORDER BY created_at DESC
             LIMIT :limit OFFSET :offset",
            $params
        );

        return [
            'datos'      => $usuarios,
            'paginacion' => [
                'total'      => $total,
                'pagina'     => $pagina,
                'por_pagina' => $porPagina,
                'paginas'    => (int)ceil($total / $porPagina),
            ],
        ];
    }

    /**
     * Obtener usuario por ID
     */
    public function obtener(int $id): array
    {
        $usuario = $this->db->fetchOne(
            'SELECT ' . self::CAMPOS_LISTADO . '
             FROM usuarios
             WHERE id = :id
             LIMIT 1',
            [':id' => $id]
        );

        if (!$usuario) {
            throw new RuntimeException("Usuario #{$id} no encontrado.", 404);
        }

        return $usuario;
    }

    /**
     * Crear nuevo usuario
     */
    public function crear(array $datos, int $adminId, string $ipAddress): array
    {
        $datos = $this->validarDatos($datos, esNuevo: true);

        // Verificar que el email no exista
        $existe = $this->db->fetchOne(
            'SELECT id FROM usuarios WHERE email = :email LIMIT 1',
            [':email' => $datos['email']]
        );

        if ($existe) {
            throw new RuntimeException('El correo electrónico ya está registrado.', 409);
        }

        $passwordHash = password_hash($datos['password'], PASSWORD_BCRYPT, ['cost' => 12]);

        $this->db->execute(
            'INSERT INTO usuarios (nombre, email, password_hash, rol, activo, created_at, updated_at)
             VALUES (:nombre, :email, :pass, :rol, :activo, NOW(), NOW())',
            [
                ':nombre' => $datos['nombre'],
                ':email'  => $datos['email'],
                ':pass'   => $passwordHash,
                ':rol'    => $datos['rol'],
                ':activo' => 1,
            ]
        );

        $nuevoId = (int)$this->db->lastInsertId();

        $this->auditLog($adminId, $nuevoId, 'INSERT', $ipAddress, [], $datos);

        return $this->obtener($nuevoId);
    }

    /**
     * Actualizar usuario existente
     */
    public function actualizar(int $id, array $datos, int $adminId, string $ipAddress): array
    {
        $antes  = $this->obtener($id);  // Lanza 404 si no existe
        $datos  = $this->validarDatos($datos, esNuevo: false);

        // Verificar email único (excluyendo el propio)
        if (isset($datos['email'])) {
            $existe = $this->db->fetchOne(
                'SELECT id FROM usuarios WHERE email = :email AND id != :id LIMIT 1',
                [':email' => $datos['email'], ':id' => $id]
            );
            if ($existe) {
                throw new RuntimeException('El correo electrónico ya está registrado.', 409);
            }
        }

        $campos = [];
        $params = [':id' => $id];

        $camposPermitidos = ['nombre', 'email', 'rol', 'activo'];
        foreach ($camposPermitidos as $campo) {
            if (array_key_exists($campo, $datos)) {
                $campos[]         = "{$campo} = :{$campo}";
                $params[":{$campo}"] = $datos[$campo];
            }
        }

        // Cambio de contraseña (opcional en actualización)
        if (!empty($datos['password'])) {
            $campos[]         = 'password_hash = :pass';
            $params[':pass']  = password_hash($datos['password'], PASSWORD_BCRYPT, ['cost' => 12]);
        }

        if (empty($campos)) {
            throw new RuntimeException('No hay campos válidos para actualizar.', 400);
        }

        $campos[] = 'updated_at = NOW()';

        $this->db->execute(
            'UPDATE usuarios SET ' . implode(', ', $campos) . ' WHERE id = :id',
            $params
        );

        $despues = $this->obtener($id);
        $this->auditLog($adminId, $id, 'UPDATE', $ipAddress, $antes, $despues);

        return $despues;
    }

    /**
     * Desactivar usuario (soft delete — nunca borrar usuarios)
     */
    public function desactivar(int $id, int $adminId, string $ipAddress): void
    {
        if ($id === $adminId) {
            throw new RuntimeException('No puede desactivar su propia cuenta.', 403);
        }

        $antes = $this->obtener($id);

        $this->db->execute(
            'UPDATE usuarios SET activo = 0, updated_at = NOW() WHERE id = :id',
            [':id' => $id]
        );

        $this->auditLog($adminId, $id, 'DELETE', $ipAddress, $antes, ['activo' => 0]);
    }

    /**
     * Cambiar contraseña (autogestionado por el propio usuario)
     */
    public function cambiarPassword(
        int    $userId,
        string $passwordActual,
        string $passwordNueva,
        string $ipAddress
    ): void {
        $usuario = $this->db->fetchOne(
            'SELECT id, password_hash FROM usuarios WHERE id = :id AND activo = 1',
            [':id' => $userId]
        );

        if (!$usuario || !password_verify($passwordActual, $usuario['password_hash'])) {
            throw new RuntimeException('Contraseña actual incorrecta.', 401);
        }

        $this->validarFortalezaPassword($passwordNueva);

        $this->db->execute(
            'UPDATE usuarios SET password_hash = :hash, updated_at = NOW() WHERE id = :id',
            [
                ':hash' => password_hash($passwordNueva, PASSWORD_BCRYPT, ['cost' => 12]),
                ':id'   => $userId,
            ]
        );

        $this->auditLog($userId, $userId, 'CAMBIO_PASSWORD', $ipAddress, [], []);
    }

    // ─────────────────────────────────────────────────────────
    // Validación y auditoría
    // ─────────────────────────────────────────────────────────

    private function validarDatos(array $datos, bool $esNuevo): array
    {
        $resultado = [];

        // Nombre
        if ($esNuevo || array_key_exists('nombre', $datos)) {
            $nombre = trim(strip_tags((string)($datos['nombre'] ?? '')));
            if (strlen($nombre) < 3 || strlen($nombre) > 100) {
                throw new RuntimeException('El nombre debe tener entre 3 y 100 caracteres.', 422);
            }
            $resultado['nombre'] = $nombre;
        }

        // Email
        if ($esNuevo || array_key_exists('email', $datos)) {
            $email = strtolower(trim(filter_var((string)($datos['email'] ?? ''), FILTER_SANITIZE_EMAIL)));
            if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                throw new RuntimeException('Correo electrónico inválido.', 422);
            }
            $resultado['email'] = $email;
        }

        // Contraseña
        if ($esNuevo || !empty($datos['password'])) {
            $password = (string)($datos['password'] ?? '');
            $this->validarFortalezaPassword($password);
            $resultado['password'] = $password;
        }

        // Rol
        if ($esNuevo || array_key_exists('rol', $datos)) {
            $rol = (string)($datos['rol'] ?? '');
            if (!in_array($rol, self::ROLES_VALIDOS, true)) {
                throw new RuntimeException('Rol inválido. Valores: ' . implode(', ', self::ROLES_VALIDOS), 422);
            }
            $resultado['rol'] = $rol;
        }

        // Activo (solo en actualización)
        if (!$esNuevo && array_key_exists('activo', $datos)) {
            $resultado['activo'] = (int)(bool)$datos['activo'];
        }

        return $resultado;
    }

    private function validarFortalezaPassword(string $password): void
    {
        if (strlen($password) < 8) {
            throw new RuntimeException('La contraseña debe tener al menos 8 caracteres.', 422);
        }
        if (!preg_match('/[A-Z]/', $password)) {
            throw new RuntimeException('La contraseña debe contener al menos una mayúscula.', 422);
        }
        if (!preg_match('/[0-9]/', $password)) {
            throw new RuntimeException('La contraseña debe contener al menos un número.', 422);
        }
    }

    private function auditLog(int $userId, int $regId, string $accion, string $ip, array $antes, array $despues): void
    {
        try {
            // Eliminar campos sensibles del log
            unset($antes['password_hash'], $despues['password_hash'], $despues['password']);
            $this->db->execute(
                'INSERT INTO auditoria_log
                    (usuario_id, tabla_afectada, registro_id, accion, datos_antes, datos_despues, ip_address, user_agent, created_at)
                 VALUES (:uid, :tabla, :regid, :accion, :antes, :despues, :ip, :ua, NOW())',
                [
                    ':uid'     => $userId,
                    ':tabla'   => 'usuarios',
                    ':regid'   => $regId,
                    ':accion'  => $accion,
                    ':antes'   => !empty($antes)    ? json_encode($antes,   JSON_UNESCAPED_UNICODE) : null,
                    ':despues' => !empty($despues)  ? json_encode($despues, JSON_UNESCAPED_UNICODE) : null,
                    ':ip'      => $ip,
                    ':ua'      => $_SERVER['HTTP_USER_AGENT'] ?? null,
                ]
            );
        } catch (\Exception) {}
    }
}
