<?php
declare(strict_types=1);

require_once CLASSES_PATH . '/Database.php';

/**
 * ============================================================
 * FRESIL C.A. — Clase Repuesto (Modelo)
 * ─────────────────────────────────────────────────────────
 * CRUD completo del catálogo de repuestos:
 * • Búsqueda full-text por nombre/OEM/código
 * • Filtros por categoría, marca vehículo, modelo, año
 * • Búsqueda por código de barras (scanner)
 * • Gestión de compatibilidades (YMME)
 * • Cálculo automático de precios en Bs con tasa del día
 * • Generación automática de código interno
 * • Paginación eficiente con COUNT optimizado
 * ============================================================
 */
class Repuesto
{
    private Database $db;

    private const UNIDADES = ['unidad','par','juego','litro','metro','kilo','set'];

    public function __construct()
    {
        $this->db = Database::getInstance();
    }

    // ─────────────────────────────────────────────────────────
    // LISTAR con filtros y paginación
    // ─────────────────────────────────────────────────────────
    public function listar(
        int    $pagina     = 1,
        int    $porPagina  = 24,
        string $buscar     = '',
        int    $categoriaId = 0,
        int    $marcaId    = 0,
        int    $modeloId   = 0,
        int    $anio       = 0,
        int    $soloStockBajo = 0,
        int    $activo     = 1,
        string $orden      = 'created_at',
        string $dir        = 'DESC'
    ): array {
        $pagina    = max(1, $pagina);
        $porPagina = min(100, max(6, $porPagina));
        $offset    = ($pagina - 1) * $porPagina;

        // Validar columna de orden (whitelist)
        $ordenesValidos = ['nombre','precio_venta_usd','stock_actual','codigo_interno','created_at'];
        if (!in_array($orden, $ordenesValidos, true)) { $orden = 'created_at'; }
        $dir = strtoupper($dir) === 'ASC' ? 'ASC' : 'DESC';

        $joins  = '';
        $where  = ['r.activo = :activo'];
        $params = [':activo' => $activo];

        // Filtro por vehículo (YMME) — requiere JOIN con compatibilidad
        if ($marcaId > 0 || $modeloId > 0 || $anio > 0) {
            $joins .= '
                JOIN repuesto_compatibilidad rc ON rc.repuesto_id = r.id
                JOIN vehiculo_anios va ON va.id = rc.vehiculo_anio_id
                JOIN modelos_vehiculo mo ON mo.id = va.modelo_id';

            if ($marcaId > 0) {
                $where[] = 'mo.marca_id = :marca_id';
                $params[':marca_id'] = $marcaId;
            }
            if ($modeloId > 0) {
                $where[] = 'va.modelo_id = :modelo_id';
                $params[':modelo_id'] = $modeloId;
            }
            if ($anio > 0) {
                $where[] = '(va.anio_inicio <= :anio AND (va.anio_fin IS NULL OR va.anio_fin >= :anio2))';
                $params[':anio']  = $anio;
                $params[':anio2'] = $anio;
            }
        }

        // Filtro por categoría (incluye subcategorías)
        if ($categoriaId > 0) {
            $where[] = '(r.categoria_id = :cat_id OR cat.parent_id = :cat_id2)';
            $params[':cat_id']  = $categoriaId;
            $params[':cat_id2'] = $categoriaId;
            $joins .= ' JOIN categorias_repuesto cat ON cat.id = r.categoria_id';
        } else {
            $joins .= ' JOIN categorias_repuesto cat ON cat.id = r.categoria_id';
        }

        // Búsqueda de texto
        if (!empty($buscar)) {
            // Intentar FULLTEXT primero (más eficiente)
            $where[] = 'MATCH(r.nombre, r.descripcion, r.codigo_oem, r.marca_repuesto) AGAINST(:buscar IN BOOLEAN MODE)
                        OR r.codigo_interno LIKE :buscar_like
                        OR r.codigo_barras  LIKE :buscar_like2';
            $params[':buscar']       = '+' . implode('* +', array_filter(explode(' ', trim($buscar)))) . '*';
            $params[':buscar_like']  = '%' . $buscar . '%';
            $params[':buscar_like2'] = '%' . $buscar . '%';
        }

        // Filtro stock bajo
        if ($soloStockBajo) {
            $where[] = 'r.stock_actual <= r.stock_minimo';
        }

        $whereStr = implode(' AND ', $where);

        // COUNT total (sin LIMIT)
        $total = (int)$this->db->fetchOne(
            "SELECT COUNT(DISTINCT r.id) AS total
             FROM repuestos r {$joins}
             WHERE {$whereStr}",
            $params
        )['total'];

        // Query principal
        $params[':limit']  = $porPagina;
        $params[':offset'] = $offset;

        $rows = $this->db->fetchAll(
            "SELECT DISTINCT
               r.id, r.codigo_interno, r.codigo_barras, r.codigo_oem,
               r.nombre, r.descripcion, r.marca_repuesto, r.unidad_medida,
               r.precio_compra_usd, r.precio_venta_usd,
               r.precio_compra_bs,  r.precio_venta_bs,
               r.tasa_cambio, r.margen_ganancia_pct,
               r.stock_actual, r.stock_minimo, r.stock_maximo,
               r.ubicacion_almacen, r.imagen_url, r.activo,
               r.created_at, r.updated_at,
               cat.id AS categoria_id, cat.nombre AS categoria_nombre, cat.icono AS categoria_icono
             FROM repuestos r {$joins}
             WHERE {$whereStr}
             ORDER BY r.{$orden} {$dir}
             LIMIT :limit OFFSET :offset",
            $params
        );

        return [
            'datos' => $rows,
            'paginacion' => [
                'total'      => $total,
                'pagina'     => $pagina,
                'por_pagina' => $porPagina,
                'paginas'    => max(1, (int)ceil($total / $porPagina)),
            ],
        ];
    }

    // ─────────────────────────────────────────────────────────
    // OBTENER uno por ID
    // ─────────────────────────────────────────────────────────
    public function obtener(int $id): array
    {
        $rep = $this->db->fetchOne(
            "SELECT r.*,
               cat.nombre AS categoria_nombre, cat.icono AS categoria_icono,
               cat.parent_id AS categoria_parent_id
             FROM repuestos r
             JOIN categorias_repuesto cat ON cat.id = r.categoria_id
             WHERE r.id = :id
             LIMIT 1",
            [':id' => $id]
        );

        if (!$rep) {
            throw new RuntimeException("Repuesto #{$id} no encontrado.", 404);
        }

        // Compatibilidades con datos de vehículo
        $rep['compatibilidades'] = $this->db->fetchAll(
            "SELECT
               rc.id, rc.posicion, rc.notas,
               va.id AS vehiculo_anio_id, va.anio_inicio, va.anio_fin, va.motor_cc, va.combustible,
               mo.id AS modelo_id, mo.nombre AS modelo, mo.tipo,
               mv.id AS marca_id, mv.nombre AS marca
             FROM repuesto_compatibilidad rc
             JOIN vehiculo_anios   va ON va.id = rc.vehiculo_anio_id
             JOIN modelos_vehiculo mo ON mo.id = va.modelo_id
             JOIN marcas_vehiculo  mv ON mv.id = mo.marca_id
             WHERE rc.repuesto_id = :id
             ORDER BY mv.nombre, mo.nombre, va.anio_inicio",
            [':id' => $id]
        );

        return $rep;
    }

    // ─────────────────────────────────────────────────────────
    // BUSCAR POR CÓDIGO DE BARRAS (para scanner)
    // ─────────────────────────────────────────────────────────
    public function buscarPorBarcode(string $codigo): array
    {
        $codigo = trim($codigo);

        if (empty($codigo)) {
            throw new RuntimeException('Código de barras vacío.', 400);
        }

        $rep = $this->db->fetchOne(
            "SELECT r.*,
               cat.nombre AS categoria_nombre, cat.icono AS categoria_icono
             FROM repuestos r
             JOIN categorias_repuesto cat ON cat.id = r.categoria_id
             WHERE (r.codigo_barras = :cod OR r.codigo_interno = :cod2)
               AND r.activo = 1
             LIMIT 1",
            [':cod' => $codigo, ':cod2' => $codigo]
        );

        if (!$rep) {
            throw new RuntimeException("Código '{$codigo}' no encontrado en inventario.", 404);
        }

        return $rep;
    }

    // ─────────────────────────────────────────────────────────
    // CREAR repuesto
    // ─────────────────────────────────────────────────────────
    public function crear(array $datos, int $userId): array
    {
        $datos = $this->validar($datos, esNuevo: true);

        // Verificar unicidad código interno
        if ($this->existeCodigo($datos['codigo_interno'])) {
            throw new RuntimeException("El código interno '{$datos['codigo_interno']}' ya existe.", 409);
        }

        // Verificar unicidad código de barras (si se provee)
        if (!empty($datos['codigo_barras']) && $this->existeBarcode($datos['codigo_barras'])) {
            throw new RuntimeException("El código de barras '{$datos['codigo_barras']}' ya existe.", 409);
        }

        $this->db->beginTransaction();
        try {
            $this->db->execute(
                'INSERT INTO repuestos
                   (codigo_interno, codigo_barras, codigo_oem, categoria_id,
                    nombre, descripcion, marca_repuesto, unidad_medida,
                    precio_compra_usd, precio_venta_usd, precio_compra_bs, precio_venta_bs,
                    tasa_cambio, margen_ganancia_pct,
                    stock_actual, stock_minimo, stock_maximo,
                    ubicacion_almacen, imagen_url, activo, created_by,
                    created_at, updated_at)
                 VALUES
                   (:ci, :cb, :oem, :cat,
                    :nombre, :desc, :marca, :unidad,
                    :pc_usd, :pv_usd, :pc_bs, :pv_bs,
                    :tasa, :margen,
                    :stock, :smin, :smax,
                    :ubic, :img, 1, :uid,
                    NOW(), NOW())',
                $this->mapParams($datos, $userId)
            );

            $id = (int)$this->db->lastInsertId();

            // Guardar compatibilidades si vienen
            if (!empty($datos['compatibilidades'])) {
                $this->guardarCompatibilidades($id, $datos['compatibilidades']);
            }

            $this->db->commit();

            $this->auditLog($userId, $id, 'INSERT', [], $datos);

            return $this->obtener($id);

        } catch (\Throwable $e) {
            $this->db->rollback();
            throw $e;
        }
    }

    // ─────────────────────────────────────────────────────────
    // ACTUALIZAR repuesto
    // ─────────────────────────────────────────────────────────
    public function actualizar(int $id, array $datos, int $userId): array
    {
        $antes = $this->obtener($id);
        $datos = $this->validar($datos, esNuevo: false);

        // Verificar unicidad código interno (excluyendo self)
        if (isset($datos['codigo_interno'])) {
            $existe = $this->db->fetchOne(
                'SELECT id FROM repuestos WHERE codigo_interno = :ci AND id != :id LIMIT 1',
                [':ci' => $datos['codigo_interno'], ':id' => $id]
            );
            if ($existe) {
                throw new RuntimeException("Código interno '{$datos['codigo_interno']}' ya existe.", 409);
            }
        }

        // Verificar unicidad código de barras
        if (!empty($datos['codigo_barras'])) {
            $existe = $this->db->fetchOne(
                'SELECT id FROM repuestos WHERE codigo_barras = :cb AND id != :id LIMIT 1',
                [':cb' => $datos['codigo_barras'], ':id' => $id]
            );
            if ($existe) {
                throw new RuntimeException("Código de barras ya existe en otro repuesto.", 409);
            }
        }

        $this->db->beginTransaction();
        try {
            $campos = [];
            $params = [':id' => $id];

            $camposPermitidos = [
                'codigo_interno','codigo_barras','codigo_oem','categoria_id',
                'nombre','descripcion','marca_repuesto','unidad_medida',
                'precio_compra_usd','precio_venta_usd','precio_compra_bs','precio_venta_bs',
                'tasa_cambio','margen_ganancia_pct',
                'stock_actual','stock_minimo','stock_maximo',
                'ubicacion_almacen','imagen_url','activo',
            ];

            foreach ($camposPermitidos as $campo) {
                if (array_key_exists($campo, $datos)) {
                    $campos[] = "{$campo} = :{$campo}";
                    $params[":{$campo}"] = $datos[$campo];
                }
            }

            if (empty($campos)) {
                throw new RuntimeException('Sin campos para actualizar.', 400);
            }

            $campos[] = 'updated_at = NOW()';
            $this->db->execute(
                'UPDATE repuestos SET ' . implode(', ', $campos) . ' WHERE id = :id',
                $params
            );

            // Actualizar compatibilidades (si se envían)
            if (array_key_exists('compatibilidades', $datos)) {
                // Eliminar las existentes y reinsertar
                $this->db->execute(
                    'DELETE FROM repuesto_compatibilidad WHERE repuesto_id = :id',
                    [':id' => $id]
                );
                if (!empty($datos['compatibilidades'])) {
                    $this->guardarCompatibilidades($id, $datos['compatibilidades']);
                }
            }

            $this->db->commit();

            $despues = $this->obtener($id);
            $this->auditLog($userId, $id, 'UPDATE', $antes, $despues);

            return $despues;

        } catch (\Throwable $e) {
            $this->db->rollback();
            throw $e;
        }
    }

    // ─────────────────────────────────────────────────────────
    // ELIMINAR (soft delete)
    // ─────────────────────────────────────────────────────────
    public function eliminar(int $id, int $userId): void
    {
        $antes = $this->obtener($id);
        $this->db->execute(
            'UPDATE repuestos SET activo = 0, updated_at = NOW() WHERE id = :id',
            [':id' => $id]
        );
        $this->auditLog($userId, $id, 'DELETE', $antes, ['activo' => 0]);
    }

    // ─────────────────────────────────────────────────────────
    // GENERAR código interno automático
    // ─────────────────────────────────────────────────────────
    public function generarCodigo(): string
    {
        $this->db->execute(
            "UPDATE secuencias SET siguiente = siguiente + 1 WHERE nombre = 'repuesto_codigo'"
        );
        $row = $this->db->fetchOne(
            "SELECT siguiente - 1 AS num FROM secuencias WHERE nombre = 'repuesto_codigo'"
        );
        return 'FRS-' . str_pad((string)(int)$row['num'], 4, '0', STR_PAD_LEFT);
    }

    // ─────────────────────────────────────────────────────────
    // RECALCULAR precios en Bs con tasa actual
    // ─────────────────────────────────────────────────────────
    public function recalcularPreciosBs(float $tasaNueva): int
    {
        return $this->db->execute(
            'UPDATE repuestos
             SET precio_compra_bs = precio_compra_usd * :tasa,
                 precio_venta_bs  = precio_venta_usd  * :tasa2,
                 tasa_cambio      = :tasa3,
                 updated_at       = NOW()
             WHERE activo = 1',
            [':tasa' => $tasaNueva, ':tasa2' => $tasaNueva, ':tasa3' => $tasaNueva]
        );
    }

    // ─────────────────────────────────────────────────────────
    // ESTADÍSTICAS para dashboard
    // ─────────────────────────────────────────────────────────
    public function estadisticas(): array
    {
        return [
            'total'          => (int)$this->db->fetchOne('SELECT COUNT(*) AS n FROM repuestos WHERE activo=1')['n'],
            'stock_bajo'     => (int)$this->db->fetchOne('SELECT COUNT(*) AS n FROM repuestos WHERE activo=1 AND stock_actual<=stock_minimo')['n'],
            'sin_stock'      => (int)$this->db->fetchOne('SELECT COUNT(*) AS n FROM repuestos WHERE activo=1 AND stock_actual=0')['n'],
            'valor_inventario'=> (float)$this->db->fetchOne('SELECT COALESCE(SUM(precio_compra_usd * stock_actual),0) AS v FROM repuestos WHERE activo=1')['v'],
            'por_categoria'  => $this->db->fetchAll(
                'SELECT c.nombre, c.icono, COUNT(r.id) AS total
                 FROM categorias_repuesto c
                 LEFT JOIN repuestos r ON r.categoria_id = c.id AND r.activo=1
                 WHERE c.parent_id IS NULL
                 GROUP BY c.id ORDER BY total DESC LIMIT 8'
            ),
        ];
    }

    // ─────────────────────────────────────────────────────────
    // Métodos privados
    // ─────────────────────────────────────────────────────────

    private function validar(array $d, bool $esNuevo): array
    {
        $out = [];

        // Nombre
        if ($esNuevo || array_key_exists('nombre', $d)) {
            $nombre = trim(strip_tags((string)($d['nombre'] ?? '')));
            if (strlen($nombre) < 3 || strlen($nombre) > 200) {
                throw new RuntimeException('Nombre debe tener entre 3 y 200 caracteres.', 422);
            }
            $out['nombre'] = $nombre;
        }

        // Código interno
        if ($esNuevo || array_key_exists('codigo_interno', $d)) {
            $ci = strtoupper(trim((string)($d['codigo_interno'] ?? '')));
            if (empty($ci)) {
                throw new RuntimeException('El código interno es requerido.', 422);
            }
            if (!preg_match('/^[A-Z0-9\-]{3,50}$/', $ci)) {
                throw new RuntimeException('Código interno solo permite letras mayúsculas, números y guiones (3-50 chars).', 422);
            }
            $out['codigo_interno'] = $ci;
        }

        // Código de barras (opcional)
        if (array_key_exists('codigo_barras', $d)) {
            $cb = trim((string)($d['codigo_barras'] ?? ''));
            $out['codigo_barras'] = !empty($cb) ? $cb : null;
        }

        // Código OEM (opcional)
        if (array_key_exists('codigo_oem', $d)) {
            $out['codigo_oem'] = !empty($d['codigo_oem']) ? trim(substr((string)$d['codigo_oem'], 0, 100)) : null;
        }

        // Categoría
        if ($esNuevo || array_key_exists('categoria_id', $d)) {
            $catId = (int)($d['categoria_id'] ?? 0);
            if ($catId <= 0) { throw new RuntimeException('Categoría inválida.', 422); }
            $cat = $this->db->fetchOne('SELECT id FROM categorias_repuesto WHERE id=:id AND activo=1', [':id'=>$catId]);
            if (!$cat) { throw new RuntimeException('Categoría no encontrada.', 422); }
            $out['categoria_id'] = $catId;
        }

        // Campos opcionales de texto
        foreach (['descripcion','marca_repuesto','ubicacion_almacen','imagen_url'] as $campo) {
            if (array_key_exists($campo, $d)) {
                $val = trim(strip_tags((string)($d[$campo] ?? '')));
                $out[$campo] = !empty($val) ? $val : null;
            }
        }

        // Unidad de medida
        if ($esNuevo || array_key_exists('unidad_medida', $d)) {
            $um = (string)($d['unidad_medida'] ?? 'unidad');
            if (!in_array($um, self::UNIDADES, true)) {
                throw new RuntimeException('Unidad de medida inválida.', 422);
            }
            $out['unidad_medida'] = $um;
        }

        // Precios USD
        foreach (['precio_compra_usd','precio_venta_usd'] as $campo) {
            if ($esNuevo || array_key_exists($campo, $d)) {
                $val = (float)($d[$campo] ?? 0);
                if ($val < 0) { throw new RuntimeException("El campo {$campo} no puede ser negativo.", 422); }
                $out[$campo] = round($val, 4);
            }
        }

        // Tasa de cambio
        if ($esNuevo || array_key_exists('tasa_cambio', $d)) {
            $tasa = (float)($d['tasa_cambio'] ?? 0);
            if ($tasa <= 0) {
                // Obtener tasa del día automáticamente
                $t = $this->db->fetchOne(
                    'SELECT tasa_usada FROM tasas_cambio ORDER BY fecha DESC LIMIT 1'
                );
                $tasa = $t ? (float)$t['tasa_usada'] : 1.0;
            }
            $out['tasa_cambio'] = round($tasa, 4);
        }

        // Precios Bs (calcular si no vienen)
        if (isset($out['precio_compra_usd']) && isset($out['tasa_cambio'])) {
            $out['precio_compra_bs'] = round($out['precio_compra_usd'] * $out['tasa_cambio'], 4);
        }
        if (isset($out['precio_venta_usd']) && isset($out['tasa_cambio'])) {
            $out['precio_venta_bs'] = round($out['precio_venta_usd'] * $out['tasa_cambio'], 4);
        }
        // Permitir sobreescribir con valores explícitos
        foreach (['precio_compra_bs','precio_venta_bs'] as $campo) {
            if (array_key_exists($campo, $d)) {
                $out[$campo] = round((float)($d[$campo] ?? 0), 4);
            }
        }

        // Margen de ganancia
        if (array_key_exists('margen_ganancia_pct', $d)) {
            $out['margen_ganancia_pct'] = round((float)($d['margen_ganancia_pct'] ?? 0), 2);
        }

        // Stock
        foreach (['stock_actual','stock_minimo'] as $campo) {
            if ($esNuevo || array_key_exists($campo, $d)) {
                $out[$campo] = max(0, (int)($d[$campo] ?? 0));
            }
        }
        if (array_key_exists('stock_maximo', $d)) {
            $sm = $d['stock_maximo'];
            $out['stock_maximo'] = ($sm !== null && $sm !== '') ? max(0, (int)$sm) : null;
        }

        // Activo
        if (array_key_exists('activo', $d)) {
            $out['activo'] = (int)(bool)$d['activo'];
        }

        // Compatibilidades (array)
        if (array_key_exists('compatibilidades', $d)) {
            $out['compatibilidades'] = is_array($d['compatibilidades']) ? $d['compatibilidades'] : [];
        }

        return $out;
    }

    private function guardarCompatibilidades(int $repuestoId, array $compats): void
    {
        foreach ($compats as $c) {
            $vaId = (int)($c['vehiculo_anio_id'] ?? 0);
            if ($vaId <= 0) { continue; }

            // Verificar que existe el vehiculo_anio
            $va = $this->db->fetchOne(
                'SELECT id FROM vehiculo_anios WHERE id = :id LIMIT 1', [':id' => $vaId]
            );
            if (!$va) { continue; }

            $posicion = !empty($c['posicion']) ? trim(substr((string)$c['posicion'], 0, 50)) : null;
            $notas    = !empty($c['notas'])    ? trim(substr((string)$c['notas'],    0, 255)) : null;

            // INSERT IGNORE para saltar duplicados
            $this->db->execute(
                'INSERT IGNORE INTO repuesto_compatibilidad
                    (repuesto_id, vehiculo_anio_id, posicion, notas)
                 VALUES (:rid, :vid, :pos, :nota)',
                [':rid'=>$repuestoId, ':vid'=>$vaId, ':pos'=>$posicion, ':nota'=>$notas]
            );
        }
    }

    private function mapParams(array $d, int $userId): array
    {
        return [
            ':ci'     => $d['codigo_interno'],
            ':cb'     => $d['codigo_barras']   ?? null,
            ':oem'    => $d['codigo_oem']       ?? null,
            ':cat'    => $d['categoria_id'],
            ':nombre' => $d['nombre'],
            ':desc'   => $d['descripcion']      ?? null,
            ':marca'  => $d['marca_repuesto']   ?? null,
            ':unidad' => $d['unidad_medida']    ?? 'unidad',
            ':pc_usd' => $d['precio_compra_usd']?? 0,
            ':pv_usd' => $d['precio_venta_usd'] ?? 0,
            ':pc_bs'  => $d['precio_compra_bs'] ?? 0,
            ':pv_bs'  => $d['precio_venta_bs']  ?? 0,
            ':tasa'   => $d['tasa_cambio']       ?? 0,
            ':margen' => $d['margen_ganancia_pct']?? 0,
            ':stock'  => $d['stock_actual']      ?? 0,
            ':smin'   => $d['stock_minimo']      ?? 1,
            ':smax'   => $d['stock_maximo']      ?? null,
            ':ubic'   => $d['ubicacion_almacen'] ?? null,
            ':img'    => $d['imagen_url']        ?? null,
            ':uid'    => $userId,
        ];
    }

    private function existeCodigo(string $codigo): bool
    {
        return (bool)$this->db->fetchOne(
            'SELECT id FROM repuestos WHERE codigo_interno = :ci LIMIT 1',
            [':ci' => $codigo]
        );
    }

    private function existeBarcode(string $barcode): bool
    {
        return (bool)$this->db->fetchOne(
            'SELECT id FROM repuestos WHERE codigo_barras = :cb LIMIT 1',
            [':cb' => $barcode]
        );
    }

    private function auditLog(int $userId, int $id, string $accion, array $before, array $after): void
    {
        try {
            $this->db->execute(
                'INSERT INTO auditoria_log
                    (usuario_id,tabla_afectada,registro_id,accion,datos_antes,datos_despues,ip_address,user_agent,created_at)
                 VALUES (:uid,:tabla,:regid,:accion,:antes,:despues,:ip,:ua,NOW())',
                [
                    ':uid'     => $userId,
                    ':tabla'   => 'repuestos',
                    ':regid'   => $id,
                    ':accion'  => $accion,
                    ':antes'   => !empty($before) ? json_encode($before, JSON_UNESCAPED_UNICODE) : null,
                    ':despues' => !empty($after)  ? json_encode($after,  JSON_UNESCAPED_UNICODE) : null,
                    ':ip'      => $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0',
                    ':ua'      => $_SERVER['HTTP_USER_AGENT'] ?? null,
                ]
            );
        } catch (\Exception) {}
    }
}
