<?php
declare(strict_types=1);

/**
 * ============================================================
 * FRESIL C.A. — Clase JWT (JSON Web Token)
 * ─────────────────────────────────────────────────────────
 * Implementación HS256 sin dependencias externas.
 * • Firma HMAC-SHA256
 * • Verificación de expiración (exp)
 * • Verificación de emisión mínima (nbf)
 * • Protección contra timing attacks en comparación
 * ============================================================
 */
class JWT
{
    /**
     * Generar un token JWT firmado
     *
     * @param array $payload Datos a incluir en el token
     * @param int   $expireMinutes Minutos hasta expiración
     * @return string Token JWT
     */
    public static function generate(array $payload, int $expireMinutes = JWT_EXPIRE_MINUTES): string
    {
        $now = time();

        $header = [
            'typ' => 'JWT',
            'alg' => 'HS256',
        ];

        $payload = array_merge($payload, [
            'iat' => $now,
            'nbf' => $now,
            'exp' => $now + ($expireMinutes * 60),
            'iss' => 'fresil-inventario',
        ]);

        $headerEncoded  = self::base64UrlEncode(json_encode($header,  JSON_THROW_ON_ERROR));
        $payloadEncoded = self::base64UrlEncode(json_encode($payload, JSON_THROW_ON_ERROR));

        $signature = self::sign($headerEncoded . '.' . $payloadEncoded);

        return $headerEncoded . '.' . $payloadEncoded . '.' . $signature;
    }

    /**
     * Validar y decodificar un token JWT
     *
     * @param string $token Token JWT a verificar
     * @return array Payload decodificado
     * @throws RuntimeException Si el token es inválido
     */
    public static function validate(string $token): array
    {
        $parts = explode('.', $token);

        if (count($parts) !== 3) {
            throw new RuntimeException('Token JWT malformado.', 401);
        }

        [$headerEncoded, $payloadEncoded, $signatureProvided] = $parts;

        // Verificar firma (protección contra timing attacks con hash_equals)
        $expectedSignature = self::sign($headerEncoded . '.' . $payloadEncoded);

        if (!hash_equals($expectedSignature, $signatureProvided)) {
            throw new RuntimeException('Firma del token JWT inválida.', 401);
        }

        // Decodificar payload
        $payload = json_decode(
            self::base64UrlDecode($payloadEncoded),
            associative: true,
            flags: JSON_THROW_ON_ERROR
        );

        $now = time();

        // Verificar expiración
        if (isset($payload['exp']) && $now > $payload['exp']) {
            throw new RuntimeException('Token JWT expirado.', 401);
        }

        // Verificar not-before
        if (isset($payload['nbf']) && $now < $payload['nbf']) {
            throw new RuntimeException('Token JWT aún no es válido.', 401);
        }

        return $payload;
    }

    /**
     * Generar token de refresco (opaco, 64 bytes hex)
     */
    public static function generateRefreshToken(): string
    {
        return bin2hex(random_bytes(64));
    }

    /**
     * Firmar el contenido con HMAC-SHA256
     */
    private static function sign(string $data): string
    {
        return self::base64UrlEncode(
            hash_hmac('sha256', $data, JWT_SECRET, binary: true)
        );
    }

    /**
     * Codificación Base64 URL-safe (RFC 4648)
     */
    private static function base64UrlEncode(string $data): string
    {
        return rtrim(strtr(base64_encode($data), '+/', '-_'), '=');
    }

    /**
     * Decodificación Base64 URL-safe
     */
    private static function base64UrlDecode(string $data): string
    {
        $decoded = base64_decode(strtr($data, '-_', '+/'), strict: false);

        if ($decoded === false) {
            throw new RuntimeException('Error decodificando token Base64.', 401);
        }

        return $decoded;
    }
}
